/* $Id: testsbst.c,v 1.6 1999/03/10 22:27:03 ericb Exp $ */
/* Copyright (C) 1995 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Chris Sutton */
                     
#include "sema.h"

#define MEM_CTL_IRQC		(0x2003L * 4)
#define VXIIC_BOOT_REG		(0x200AL * 4)	/* VXI A24 mod address */
#define VXIIC_CMD_REG		(0x200BL * 4)	

#define HOSTB_SEM		(0x2324L * 4)
#define HOSTB_IVR		(0x2330L * 4)

#define MEM_CTL_IRQ		(0x2008L * 4)

#define CLOCK_TRIG_SETUP	(0x2100L * 4)
#define CLOCK_TRIG_TTL		(0x2101L * 4)
#define CLOCK_TRIG_STATE	(0x2102L * 4)
#define CLOCK_TRIG_MDIV		(0x2108L * 4)
#define CLOCK_TRIG_NDIV		(0x2109L * 4)
#define CLOCK_TRIG_PDIV		(0x210aL * 4)
#define CLOCK_TRIG_QDIV		(0x210bL * 4)
#define CLOCK_TRIG_RDIV		(0x210cL * 4)

#define CLOCK_TRIG_STATUS	(0x2106L * 4)

/*
 *************************************************************************
 e1432_test_vxi()
 
 Test the VXI interface.
 
 *************************************************************************
 */
SHORTSIZ16
i1432_test_vxi( E1432_MODULE_LIST_NODE *mn )
{
    SHORTSIZ16 err = 0;
    USHORTSIZ16 wVal, wVal2;
/*
    USHORTSIZ16 model;
*/
    LONGSIZ32 lVal;
    
    DIAG_PRINTF(1, ("Testing E1432 VXI interface.\n"));

    /* formerly done in this test:
       - read the device type register of the slot0 controller
	 no longer done
       - read E1432 ID register
	 now handled in i1432_sicl_setup1()
       - read Device type register, Check Device Type register
	 now handled in i1432_sema_present
    */

    DIAG_PRINTF(2, ("Writing to E1432 VXI registers.\n"));

    /* write to control register  */
    err =i1432_direct_write_register(mn, E1432_CONTROL_REG,
				     (SHORTSIZ16) 0x8002);
    if ( err )
    {
	i1432_print_acc_fail(1, err, "writing to control register");
	DIAG_PRINTF(1, ("  This was the first attempt to write to a E1432"
	  " register."));
	DIAG_PRINTF(1, ("  Suspect:  logic near VXI interface U780.\n"));
	DIAG_PRINTF(0, ("  VXI test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);        
    }                                                                            

    DIAG_PRINTF(1, ("Reseting E1432.\n"));

    /* Assert reset */
    err = i1432_direct_write_register(mn, E1432_CONTROL_REG,
				      (SHORTSIZ16) 0x8003);
    i1432_pause(0.1);
    /* Release reset */
    if ( ! err )
    {
        err = i1432_direct_write_register(mn, E1432_CONTROL_REG,
					  (SHORTSIZ16) 0x8002);
    }
    if ( err )
    {
	i1432_print_acc_fail(1, err,
	  "trying to reset the E1432 by writing to its\n  CONTROL register");
	DIAG_PRINTF(0, ("  VXI test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);        
    }

    /* If the E1432 B-bus is hung waiting for TA, it may not un-reset.
       Doing a peek seems to help. */
    /* Help release reset */
    (void) i1432_direct_read_register(mn, 0x10, (SHORTSIZ16 *)&wVal);
    /* Assert reset */
    (void) i1432_direct_write_register(mn, E1432_CONTROL_REG,
				       (SHORTSIZ16) 0x8003);
    i1432_pause(0.1);
    /* Release reset */
    (void) i1432_direct_write_register(mn, E1432_CONTROL_REG,
				       (SHORTSIZ16) 0x8002);
    i1432_pause(0.5);

    /* ( if (diag_stop_testing()) return DIAG_ABORT; */
    /* if (diagFlags & DIAG_FLAGS_ALLOW_ROM_BOOT) */
    	i1432_pause(1.0); 
    /* ( if (diag_stop_testing()) return DIAG_ABORT; */

    /* read STATUS register  */
    err = i1432_direct_read_register(mn, E1432_STATUS_REG, (SHORTSIZ16 *)&wVal);
    if ( err )
    {
	i1432_print_acc_fail(1, err,
	  "when attemting to read the E1432 STATUS register");
	DIAG_PRINTF(1, ("\n"));
	DIAG_PRINTF(1, ("  Suspect:  VXI interface U780, its ROM U707. \n"));
	DIAG_PRINTF(0, ("  VXI test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);        
    }                                                                            
    
    /* check STATUS register response */
    if ((wVal & 0xF010)!= 0xC010) {
	DIAG_PRINTF(1, ("  The E1432 STATUS register returned the wrong value"
	  " after reset.\n"));
	DIAG_PRINTF(1, ("  Expected 0xC-1-, got 0x%04X\n", (int)wVal));
	if (!(wVal & E1432_STATUS_HW_OK)) {
	    DIAG_PRINTF(1, ("  The HW_OK bit was 0, expected 1."
	      "  This indicates the PROGn line\n"));
	    DIAG_PRINTF(1, ("  was held low by a Xilinx because it did"
	      " not load. \n"));                        
	    DIAG_PRINTF(1, ("  Suspect Xilinx parts and thier ROMS: U708,"
	      " U705, U380, U381.\n"));                        
	}
	if (!(wVal & E1432_STATUS_PASSED)) {
	    DIAG_PRINTF(1, ("  The PASSED bit was 0, expected 1.\n"));
	    DIAG_PRINTF(1, ("  It should have been set by the 96000 comming"
	      " out of reset.\n"));
	    DIAG_PRINTF(1, ("  Suspect Memory Control Xilinx U708, 96002_RSTn"
	      " line, One shot U574.\n"));                        
	    DIAG_PRINTF(1, ("  If dip switchs S2 are set to boot from ROM"
	      " suspect 96000 and boot ROM U728.\n"));
	}
	DIAG_PRINTF(0, ("  VXI test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);        
    } 

    /* Check Device type and ID register from A24 space */
/* ifdef out for now, fix later */
#if 0
    /* try to make work with an new i1432_direct_read_register_a()
    err = i1432_direct_read_register_a(mn, E1432_A24, E1432_ID_REG, &wVal);
    if ( err )
    {
	i1432_print_acc_fail(1, err,
	  "attemting to read the E1432 ID register");
	DIAG_PRINTF(1, ("  A16 read was ok, A24 fails.\n"));
	DIAG_PRINTF(1, ("\n"));
	DIAG_PRINTF(1, ("  Suspect:  VXI interface U780. \n"));
	DIAG_PRINTF(0, ("  VXI test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);
    }
    if (wVal != 0xCFFF)
    {
	DIAG_PRINTF(1, ("  An A24 read of the ID register returned a bad"
	  " value.  A16 read was OK.\n"));
	DIAG_PRINTF(1, ("  Expected 0xcfff, got 0x%4.4x\n",
	  wVal));
	DIAG_PRINTF(1, ("  Suspect:  VXI interface U780. \n"));
	DIAG_PRINTF(0, ("  VXI test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);
    }

    err = i1432_direct_read_register_a(mn, E1432_A24, E1432_DEVICE_TYPE_REG,
      &wVal);
    if ( err )
    {
	i1432_print_acc_fail(1, err,
	  "attemting to read the E1432 DEVICE TYPE register");
	DIAG_PRINTF(1, ("  The ID register read OK.\n"));
	DIAG_PRINTF(1, ("  The A16 read of the DEVICE TYPE register"
	  " was OK.\n"));
	DIAG_PRINTF(1, ("\n"));
	DIAG_PRINTF(1, ("  Suspect:  VXI interface U780, its ROM U707. \n"));
	DIAG_PRINTF(0, ("  VXI test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);
    }
    model = wVal & 0x0fff;
    a24size = wVal & 0xf000;
    if ((unsigned long) model < 0x200 || (unsigned long) model > 0x203)
    {
	DIAG_PRINTF(1, ("  The E1432 DEVICE TYPE register returned the wrong"
	  " model\n"));
	DIAG_PRINTF(1, ("  code when read from A24.  Expected 0x200 or 0x201"
	  " or 0x202\n"));
	DIAG_PRINTF(1, ("  or 0x203, got 0x%04X\n", (int) model));
	DIAG_PRINTF(1, ("\n"));
	DIAG_PRINTF(1, ("  Suspect:  VXI interface U780, its ROM U707. \n"));
	DIAG_PRINTF(0, ("  VXI test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);
    }
    if (a24size != 0x3000 && a24size != 0x5000)
    {
	DIAG_PRINTF(1, ("  The E1432 DEVICE TYPE register returned the"
	  " wrong value\n"));
	DIAG_PRINTF(1, ("  for A24 size when read from A24.  Expected 0x3000"
	  " or\n"));
	DIAG_PRINTF(1, ("  0x5000, got 0x%04X\n", (int) a24size));
	DIAG_PRINTF(0, ("  VXI test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);
    }

    if (mn->a24_256k && (a24size != 0x5000))
    {
	DIAG_PRINTF(1, ("  The E1432 DEVICE TYPE register returned a"
	  " different\n"));
	DIAG_PRINTF(1, ("  value when read from A24 than from A16.\n"));
	DIAG_PRINTF(1, ("  Expected 0x520x, got 0x%04X\n", (int) wVal));
    }
    if (!mn->a24_256k && ((wVal & 0xf000) != 0x3000))
    {
	DIAG_PRINTF(1, ("  The E1432 DEVICE TYPE register returned a"
	  " different\n"));
	DIAG_PRINTF(1, ("  value when read from A24 than from A16.\n"));
	DIAG_PRINTF(1, ("  Expected 0x320x, got 0x%04X\n", (int) wVal));
    }
#endif
# if 0 /* not doing the no boot option any more */
    if ((OVal & 0x0FFF) == 0x0201 &&
	!(diagFlags & DIAG_FLAGS_ALLOW_ROM_BOOT))
    {
    	DIAG_PRINTF(1, ("  Diagnostics test stopped because E1432 reports it\n"));
    	DIAG_PRINTF(1, ("  has booted from ROM.\n"));
    	DIAG_PRINTF(1, ("  Check:  DIP switch S2: switches 1,2,3 should be Open.\n"));
	DIAG_PRINTF(0, ("  VXI test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);
    }
#endif
    /* check page map register */
    err = i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, E1432_BASE_PAGE);
    if ( ! err )
    {
        err = i1432_direct_read_register(mn, E1432_PAGE_MAP_REG,
	  (SHORTSIZ16 *)&wVal);
    }
    if ( err )
    {
	i1432_print_acc_fail(1, err, "reading the page map register");
	DIAG_PRINTF(0, ("  VXI test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);        
    }                                                                            
    if (wVal!=E1432_BASE_PAGE) {
    	DIAG_PRINTF(1, ("  Bad value from PAGE MAP reg.\n"));
	DIAG_PRINTF(1, ("  Expected 0x0000, got 0x%04X\n", (int)wVal));
	DIAG_PRINTF(0, ("  VXI test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);        
    }
    if (mn->a24_256k)
	wVal2 = 0x03ff;
    else
	wVal2 = 0x00ff;
    (void) i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, wVal2);
    (void) i1432_direct_read_register(mn, E1432_PAGE_MAP_REG,
      (SHORTSIZ16 *)&wVal);
    if (wVal!=wVal2) {
	DIAG_PRINTF(1, ("  Bad value from PAGE MAP reg.\n"));
	DIAG_PRINTF(1, ("  Expected 0x%04X, got 0x%04X\n",
	  (int) wVal2, (int)wVal));
	DIAG_PRINTF(0, ("  VXI test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);        
    }

    /* Check 32-bit I/O from A24 space */
    i1432_r32(mn, &err, E1432_MEAS_STATE_REG, &lVal);
    if ( err )
    {
	i1432_print_acc_fail(1, err,
	  "doing 32-bit A24 read of the B-bus Static RAM");
	DIAG_PRINTF(1, ("  Failed on the first B-bus access.\n"));
	DIAG_PRINTF(1, ("  Suspect:  VXI interface U780, B-bus decode EPLD"
	  " U740, the 96000,\n"));
	DIAG_PRINTF(1, ("  signals like bVRn, VGn, bTAn. \n"));
	DIAG_PRINTF(0, ("  VXI test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);
    }
    i1432_w32(mn, &err, E1432_MEAS_STATE_REG, 0x12345678L);
    if ( err )
    {
	i1432_print_acc_fail(1, err, "doing 32-bit A24 write");
	DIAG_PRINTF(1, ("  An earlier 32-bit A24 read was OK.\n"));
	DIAG_PRINTF(0, ("  VXI test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);
    }
    i1432_r32(mn, &err, E1432_MEAS_STATE_REG, &lVal);
    if ( err )
    {
	i1432_print_acc_fail(1, err, "doing 32-bit A24 read");
	DIAG_PRINTF(1, ("  Earlier 32-bit A24 reads and writes were OK.\n"));
	DIAG_PRINTF(0, ("  VXI test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);
    }
    /* don't check for correct value here -- let the B SRAM test do that */

    return 0;	        	
}


/*
 *************************************************************************
 e1432_test_vxi_dsp()
 
 Write a pattern to and A24 address that maps
 thru the VXI IC's "fixed DSP bus window" to the
 b-bus, back to the VXI IC's DSP-accessible
 registers.  Expect the values to show up
 in status bits and DEVICE TYPE reg.
 
 *************************************************************************
 */

SHORTSIZ16
i1432_test_vxi_dsp( E1432_MODULE_LIST_NODE *mn )
{
    SHORTSIZ16 err = 0;
    SHORTSIZ16 err_count = 0;
    SHORTSIZ16 readVal;
    LONGSIZ32 readValL;
    	
    DIAG_PRINTF(1, ("Testing VXI IC Boot register from B-Bus.\n"));
    
    /* set page map register to base page */
    err =i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, E1432_BASE_PAGE);
    if ( err )
    {
	i1432_print_acc_fail(1, err, "writing to page map register");
	DIAG_PRINTF(0, ("  VXI test failed.\n"));
        return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);
    }

    i1432_w32(mn, &err, VXIIC_BOOT_REG, 0x5555L);
    if ( err ) {
	i1432_print_acc_fail(1, err, "using the B-bus to write to the BOOT"
	  " register\n  in the VXI interface IC U780");
	DIAG_PRINTF(0, ("  VXI test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);        
    }                                                                            
    /* read and check Device type register */
    (void) i1432_direct_read_register(mn, E1432_DEVICE_TYPE_REG, &readVal);
    if ((readVal & 0x0fff) != 0x0555) {
	DIAG_PRINTF(1, ("  Expected 0x3555 or 0x5555 got 0x%04X in DEVICE"
	  " TYPE reg.\n", (int)readVal));
	err_count++;        
    } 
    /* read and check status register */
    (void) i1432_direct_read_register(mn, E1432_STATUS_REG, &readVal);
    if ((readVal & 0x0308) != 0x0208) {
	DIAG_PRINTF(1, ("  Wrote 0x5555 got 0x%04X in STATUS reg.\n",
	  (int)readVal));
	err_count++;        
    } 

    /* do it again with inverted pattern */
    i1432_w32(mn, &err, VXIIC_BOOT_REG, 0xAAAAL);
    (void) i1432_direct_read_register(mn, E1432_DEVICE_TYPE_REG, &readVal);
    if ((readVal & 0x0fff) != 0x0aaa) {
	DIAG_PRINTF(1, ("  Expected 0x3aaa or 0x5aaa got 0x%04X in DEVICE TYPE"
	  " reg.\n", (int)readVal));
	err_count++;        
    } 
    (void) i1432_direct_read_register(mn, E1432_STATUS_REG, &readVal);
    if ((readVal & 0x0308) != 0x0100) {
	DIAG_PRINTF(1, ("  Wrote 0xAAAA got 0x%04X in STATUS reg.\n",
	  (int)readVal));
	err_count++;        
    } 
    /* clean up */    
    i1432_w32(mn, &err, VXIIC_BOOT_REG, 0x0200L);
    if (err_count) {
	DIAG_PRINTF(1, ("  Diagnostics failed.  Bad value in the STATUS or"
	  " DEVICE TYPE \n"));
	DIAG_PRINTF(1, ("  register after using the B-bus to write to the"
	  " BOOT register\n"));
	DIAG_PRINTF(1, ("  in VXI interface IC U780.\n"));
	DIAG_PRINTF(0, ("  VXI test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);        
    }

    	/**********************************************
    	 * DSP Command Register in VXI IC                     
    	 *
    	 * .. not done yet.
    	 *************************/


    	/**********************************************
    	 * Check the memory controller while
    	 * we are at it.   It should be safe to mess
    	 * with an IRQ Control register since the 
    	 * 96000 is waiting to boot or has only 
    	 * booted from rom.                  
    	 *
    	 *************************/
    i1432_w32(mn, &err, MEM_CTL_IRQC, 0x0055L);
    if ( err ) {
	i1432_print_acc_fail(1, err,
	  "using the B-bus to write to\n  the Memory Control Xilinx U708");
	DIAG_PRINTF(0, ("  VXI test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);        
    }
    i1432_r32(mn, &err, MEM_CTL_IRQC, &readValL);
    readValL &= 0x00ff;
    if (readValL != 0x0055) {
	DIAG_PRINTF(1, ("  Wrote 0x0055 got 0x%04X.\n", (int)readValL));
	err_count++;        
    } 
    i1432_w32(mn, &err, MEM_CTL_IRQC, 0x00AAL);
    i1432_r32(mn, &err, MEM_CTL_IRQC, &readValL);
    readValL &= 0x00ff;
    if (readValL != 0x00AA) {
	DIAG_PRINTF(1, ("  Wrote 0x00AA got 0x%04X.\n", (int)readValL));
	err_count++;        
    } 
    if (err_count && (readValL == 0xff)) {
	DIAG_PRINTF(1, ("  Diagnostics failed.  Bad value read back from Memory Control\n"));
	DIAG_PRINTF(1, ("  Xilinx U708. Suspect ROM U705.\n"));
	DIAG_PRINTF(0, ("  VXI test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);        
    }
    if (err_count) {
	DIAG_PRINTF(1, ("  Diagnostics failed.  Bad value read back from Memory Control\n"));
	DIAG_PRINTF(1, ("  Xilinx U708.  Susupect B-Data lines.\n"));
	DIAG_PRINTF(0, ("  VXI test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);        
    }
    i1432_w32(mn, &err, MEM_CTL_IRQC, 0L);
    
    return 0;	        	
}


/*
 *************************************************************************
 e1432_test_96k_hostport()
 
 Write a pattern to and A24 address that maps
 to the 96000 host port SEM and IVR registers. 
 
 *************************************************************************
 */


SHORTSIZ16
i1432_test_96k_hostport( E1432_MODULE_LIST_NODE *mn )
{
    SHORTSIZ16 err = 0;
    SHORTSIZ16 err_count = 0;
    LONGSIZ32 readVal;
    
    /* set page map register to base page */
    err =i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, E1432_BASE_PAGE);
    if ( err )
    {
	i1432_print_acc_fail(1, err, "writing to page map register");
	DIAG_PRINTF(0, ("  96000 Host Port test failed.\n"));
        return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);
    }

    DIAG_PRINTF(1, ("Testing 96000 Host Port from B-Bus.\n"));

    i1432_w32(mn, &err, HOSTB_SEM, 0x5555L);
    i1432_w32(mn, &err, HOSTB_IVR, 0x00AAL);
    if ( err ) {
	i1432_print_acc_fail(1, err,
	  "using the B-bus to write to the 96000 Host Port");
	DIAG_PRINTF(0, ("  96000 Host Port test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);
    }   
                                                                             
    /* read and check  */
    i1432_r32(mn, &err, HOSTB_SEM, &readVal);
    if ( err ) {
	i1432_print_acc_fail(1, err,
	  "using the B-bus to read from the 96000 Host Port");
	DIAG_PRINTF(0, ("  96000 Host Port test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);
    }   
    if (readVal  != 0x5555L) {
	DIAG_PRINTF(1, ("  Expected 0x00005555 got 0x%08lX in SEMA register.\n",
	  readVal));
	err_count++;        
    }             
    i1432_r32(mn, &err, HOSTB_IVR, &readVal);
    if (readVal  != 0x00AAL) {
	DIAG_PRINTF(1, ("  Expected 0x000000AA got 0x%08lX in IVR register.\n",
	  readVal));
	err_count++;        
    }             
    
    /* do it again with inverted pattern */
    i1432_w32(mn, &err, HOSTB_SEM, 0xAAAAL);
    i1432_w32(mn, &err, HOSTB_IVR, 0x0055L);
    i1432_r32(mn, &err, HOSTB_SEM, &readVal);
    if (readVal  != 0xAAAAL) {
	DIAG_PRINTF(1, ("  Expected 0x0000AAAA got 0x%08lX in SEM register.\n",
	  readVal));
	err_count++;        
    } 
    i1432_r32(mn, &err, HOSTB_IVR, &readVal);
    if (readVal  != 0x0055L) {
	DIAG_PRINTF(1, ("  Expected 0x00000055 got 0x%08lX in IVR register.\n",
	  readVal));
	err_count++;        
    } 

    if (err_count) {
	DIAG_PRINTF(0, ("  96000 Host Port test failed.\n"));
	DIAG_PRINTF(1, ("  Bad data after writing and reading a host-port register in \n"));
	DIAG_PRINTF(1, ("  the 96000 via the B-bus.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);
    }

    return 0;	        	
}


/*
 *************************************************************************
 wait_for_pll_lock()
 
  Wait for the PLL to lock.  returns an error if it does not.
 
 *************************************************************************
 */

static SHORTSIZ16
wait_for_pll_lock(E1432_MODULE_LIST_NODE *mn)
{
    SHORTSIZ16 err = 0;
    FLOATSIZ64 timeSec;
    FLOATSIZ64 startTimeSec;
    FLOATSIZ64 startLockTimeSec;
    LONGSIZ32 lockBits;

    i1432_get_time( &startTimeSec );
    startLockTimeSec = startTimeSec; 
    do {
	i1432_r32(mn, &err, MEM_CTL_IRQ, &lockBits);
	if ( err ) return err;
    	i1432_get_time( &timeSec );          
    	
    	/* if unlocked reset the lock timer  */
    	if ((lockBits & 0x0008) || !(lockBits & 0x2000))
    	    startLockTimeSec = timeSec;	
    	    
    	/* if timeout report an error */
    	if (timeSec > startTimeSec + 6.0)
    	    return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);
        /* if (diag_stop_testing()) return DIAG_ABORT; */
    	/* wait for 1 sec of solid lock  */	
    } while (timeSec < startLockTimeSec + 1.0);
    
    return 0;
}

/*
 *************************************************************************
 e1432_test_pll()
 
 Test phase locked loop and other ckts by
 writing to the Clock-trigger Xilinx via an 
 A24 address that maps thru the VXI IC's 
 "fixed DSP bus window".
 
 *************************************************************************
 */


SHORTSIZ16
i1432_test_pll( E1432_MODULE_LIST_NODE *mn )
{
    SHORTSIZ16 err = 0;
    LONGSIZ32 readVal;
    
    DIAG_PRINTF(1, ("Testing PLL, Clock and Trigger.\n"));

    /* set page map register to base page */
    err =i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, E1432_BASE_PAGE);
    if ( err )
    {
	i1432_print_acc_fail(1, err, "writing to page map register");
	DIAG_PRINTF(0, ("  PLL test failed.\n"));
        return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);
    }

	/***************************
	 * check rev register in 
	 * Clock-Trig Xilinx to
	 * see if we can talk to it. 
	 **************************/
	 
    /* read and check STATUSRD register in  */
    i1432_r32(mn, &err, CLOCK_TRIG_STATUS, &readVal);
    if ( err ) {
	i1432_print_acc_fail(1, err,
	  "using the B-bus to read Clock-Trigger Xilinx U380");
	DIAG_PRINTF(1, ("  Clock-Trigger Xilinx U380."
	  "  Suspect VXI interface, not U380.\n"));
	DIAG_PRINTF(0, ("  PLL test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);        
    }   
                                                                             
    readVal &= 0x0f;                          
    
#ifdef PRODUCTION_TEST
    if (readVal  != 0x04) {
#else
    if ((readVal  < 0x03) || (readVal  == 0x0f)) {
#endif    
	DIAG_PRINTF(1, ("  Diagnostics failed."
	  "  Bad value read from Clock-Trigger STATUS \n"));
	DIAG_PRINTF(1, ("  register in U380."
	  "  Suspect bD(0:7) lines, U380.\n"));
	DIAG_PRINTF(1, ("  This is the 'revision' register,"
	  " so a new or old Xilinx ROM\n"));
	DIAG_PRINTF(1, ("  U381 will cause this failure.\n"));
	DIAG_PRINTF(1, ("  Expected 0x04 got 0x%02X in STATUS register.\n",
	  (int)readVal));
	DIAG_PRINTF(0, ("  PLL test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);        
    }             

	/***************************
	 * Program Clock-Trig and 
	 * PLL for 51.2k Sa/s.  
	 * See if PLL locks. 
	 **************************/
                                               
    /* program SETUP reg with typical setup: sema4, 24.5MHz, block...  */                                               
    i1432_w32(mn, &err, CLOCK_TRIG_SETUP, 0x003651UL);

    /* select TTLTRIG lines */                                               
    i1432_w32(mn, &err, CLOCK_TRIG_TTL, 0UL);

    /* program M, N, P, Q, R dividers */
    i1432_w32(mn, &err, CLOCK_TRIG_MDIV, 480 - 2 );
    i1432_w32(mn, &err, CLOCK_TRIG_NDIV, 384 - 2 );
    i1432_w32(mn, &err, CLOCK_TRIG_PDIV, 0x0 );	/*  /1 for sample1 */
    i1432_w32(mn, &err, CLOCK_TRIG_QDIV, 0x0c00003UL );	/*  /8 for sample2 */
    i1432_w32(mn, &err, CLOCK_TRIG_RDIV, 0x0400000UL );	/*  /2 for sample3 */
    if ( err ) {
	i1432_print_acc_fail(1, err,
	  "using the B-bus to write to Clock-Trigger Xilinx U380");
	DIAG_PRINTF(1, ("  Reading did not bus error.\n"));
	DIAG_PRINTF(0, ("  PLL test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);
    }   

    err = wait_for_pll_lock(mn);
    if ( err && err != ERR1432_DIAG_ERR_SUBST ) {
	i1432_print_acc_fail(1, err, "checking PLL lock");
	DIAG_PRINTF(0, ("  PLL test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);
    }
    if ( err ) {
	DIAG_PRINTF(1, ("  Diagnostics failed."
	  "  Phase Locked Loop failed to lock.\n"));
	DIAG_PRINTF(1, ("  Clock-Trigger and PLL circuits were programmed"
	  " to divide\n"));
	DIAG_PRINTF(1, ("  the 24.576MHz from Crystal Y300 by 480"
	  " using U300-305.\n"));
	DIAG_PRINTF(1, ("  This signal, SAMPLE0, at 51.2 kHz,"
	  " is then multiplied by 384\n"));
	DIAG_PRINTF(1, ("  to 19.6MHz using PLL circuits near U350.\n"));
	DIAG_PRINTF(1, ("  \n"));
	DIAG_PRINTF(1, ("  Check: 24.5 MHz at U300 pin 1 (MCLK).\n"));
	DIAG_PRINTF(1, ("         51.2 kHz narrow pulse at TP370"
	  " (SAMPLE0).\n"));
	DIAG_PRINTF(1, ("          +5V at TP520 ( +5V1).\n"));
	DIAG_PRINTF(1, ("         -15V at TP511 (-15V1).\n"));
	DIAG_PRINTF(1, ("         +15V at TP501 (+15V1).\n"));
	DIAG_PRINTF(1, ("         19.6 MHz at TP371 (SAMPLE1).\n"));
	DIAG_PRINTF(1, ("  Check soldering of inductors like L330."));
	DIAG_PRINTF(0, ("  PLL test failed.\n"));
	DIAG_PRINTF(1, ("\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);        
    }

	/***************************
	 * Use 10 MHz, see if it
	 * still locks. 
	 **************************/
    /* program SETUP reg 10MHz  */
    i1432_w32(mn, &err, CLOCK_TRIG_SETUP, 0x003650UL );
    /* program M dividers for about 51.2k */
    i1432_w32(mn, &err, CLOCK_TRIG_MDIV, 195 - 2 );
    err = wait_for_pll_lock(mn);
    if ( err && err != ERR1432_DIAG_ERR_SUBST ) {
	i1432_print_acc_fail(1, err, "checking 10 MHz PLL lock");
	DIAG_PRINTF(0, ("  PLL test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);
    }
    if ( err ) {
	DIAG_PRINTF(1, ("  Diagnostics failed."
	  "  Phase Locked Loop failed to lock \n"));
	DIAG_PRINTF(1, ("  when using the 10 MHz CLK10 signal from VXI"
	  " backplane.\n"));
	DIAG_PRINTF(1, ("  PLL locked OK when using 24.576 MHz from Crystal"
	  " Y300.\n"));
	DIAG_PRINTF(1, ("  \n"));
	DIAG_PRINTF(1, ("  Check: 10.0 MHz at U300 pin 1 (MCLK).\n"));
	DIAG_PRINTF(1, ("\n"));
	DIAG_PRINTF(0, ("  PLL test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);        
    }



    return 0;	        	
}
